%% JY5325多通道有限点输出示例
clc;clear;
% =========================
% 常量配置（可按需修改）
% =========================
slotNumber       = int32(0);      % 槽位索引
channels         = [0 1];         % 两个通道编号（示例：AO0 与 AO1）
updateRate       = 100000.0;       % 采样率(Hz)
samplesToUpdate  = int32(1000000);  % 有限点输出样本数

% 通道独立波形配置
% 通道0：正弦 1 kHz，振幅 2 V
waveType0        = 'sine';        % 'sine' | 'square' | 'noise'
amplitude0       = 2.0;
frequency0       = 1000.0;
% 通道1：方波 500 Hz，振幅 1.5 V
waveType1        = 'square';      % 'sine' | 'square' | 'noise'
amplitude1       = 1.5;
frequency1       = 500.0;

% JY5325 .NET程序集路径
jyDllPath='C:\SeeSharp\JYTEK\Hardware\DAQ\JY5325\Bin\JY5325.dll';
% =========================
% 载入 .NET 程序集
% =========================
try
    asms = System.AppDomain.CurrentDomain.GetAssemblies();
    found = false;
    for k = 1:numel(asms)
        if contains(char(asms(k).FullName), 'JY5325')
            found = true;
            break;
        end
    end
    if ~found
        NET.addAssembly(jyDllPath);
    end
catch
    NET.addAssembly(jyDllPath);
end
import JY5325.*

% =========================
% 为每个通道生成波形
% =========================
N = double(samplesToUpdate);
t = (0:N-1) / updateRate;

% 通道0
switch lower(waveType0)
    case 'sine'
        ch0 = amplitude0 * sin(2*pi*frequency0*t);
    case 'square'
        ch0 = amplitude0 * sign(sin(2*pi*frequency0*t)); % 50% 占空
    case 'noise'
        ch0 = amplitude0 * (2*rand(1, N) - 1);
    otherwise
        error('Unsupported waveType0: %s', waveType0);
end
% 通道1
switch lower(waveType1)
    case 'sine'
        ch1 = amplitude1 * sin(2*pi*frequency1*t);
    case 'square'
        ch1 = amplitude1 * sign(sin(2*pi*frequency1*t)); % 50% 占空
    case 'noise'
        ch1 = amplitude1 * (2*rand(1, N) - 1);
    otherwise
        error('Unsupported waveType1: %s', waveType1);
end
% 可视化（预览两个通道波形）
figure('Name', 'AO Finite Dual-Channel Output (Preview)');
subplot(2,1,1); plot(t, double(ch0)); grid on;
title('Channel 0'); xlabel('Time (s)'); ylabel('Voltage (V)');
subplot(2,1,2); plot(t, double(ch1)); grid on;
title('Channel 1'); xlabel('Time (s)'); ylabel('Voltage (V)');

% 组装为 double[,]（形状：Samples x Channels），与 C# 的二维数组一致
% 注意：JY5325 AO 写入使用列主序通道布局（参考示例：writeValue[k, i]）
data2D = NET.createArray('System.Double', N, numel(channels));
for k = 1:N
    data2D(k,1) = ch0(k);
    data2D(k,2) = ch1(k);
end

% =========================
% 建立并配置 AO 有限点任务
% =========================
aoTask = JY5325AOTask(slotNumber);
cleanupObj = onCleanup(@()safeCleanup(aoTask)); % 自动清理
% 采样时钟（内部）
aoTask.SampleClock.Source = AOSampleClockSource.Internal;
% 添加两个通道
for i = 1:numel(channels)
    aoTask.AddChannel(int32(channels(i)));
end
% 基本参数
aoTask.Mode            = AOMode.Finite;
aoTask.UpdateRate      = updateRate;
aoTask.SamplesToUpdate = samplesToUpdate;

% =========================
% 写数据并启动
% =========================
aoTask.WriteData(data2D, int32(-1)); % -1 表示无限等待
aoTask.Start();
% 轮询等待任务完成（等效 WinForms 的 Timer + WaitUntilDone(10)）
count=0;
while ~aoTask.WaitUntilDone(int32(10))
    % 打印缓冲区点数
    avaliableSamples=double(aoTask.AvaliableLenInSamples);     % 获取缓冲区点数
    fprintf(repmat('\b',1,count));
    count=fprintf('The availableSamples: %d',int32(avaliableSamples));
    pause(0.1);
end
fprintf('\n');
% 停止与清理
aoTask.Stop();
aoTask.Channels.Clear();